% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program RT.m

function RT (C_in_Os)

%% Display Setup Module

% Define display parameters

whichScreen = max(Screen('screens'));
p.ScreenDistance = 30; 	% in inches
p.ScreenHeight = 15; 	% in inches
p.ScreenGamma = 2;	% from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5;

% Open the display window and hide the mouse cursor

if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
PsychImaging('PrepareConfiguration'); 
PsychImaging('AddTask', 'General', 'FloatingPoint32BitIfPossible');   % set up a 32-bit framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange');
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');  % setup Gamma correction method
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % open a display window
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);  % set Gamma for all color channels
% HideCursor;
HideCursor;

% Get frame rate and set screen font

p.ScreenFrameRate = FrameRate(windowPtr);
Screen('TextFont', windowPtr, 'Times'); 
Screen('TextSize', windowPtr, 24);

%% Experimental Module

% Specify the stimulus
p.stimSize = 0.98;      % letter size in visual degree
p.radius = 4.12;        % radius of the annulus in degree
p.jitter = 4;           % range of location jitter in pixels
p.dispSize = [4 8 12];  % total # of items in the display
repeats = 80;           % number of trials in each 
                        % experimental condition
p.fixDuration = 0.25;
p.ITI = 1;              % time interval between trials in 
                        % seconds
keys = {'left' 'right' 'esc'};  % response keys for target 
                        % absent and present, and to break

% Compute stimulus parameters
ppd = pi/180 * p.ScreenDistance / p.ScreenHeight * p.ScreenRect(4);        % pixels per degree
m = round(p.stimSize * ppd);  % letter size in pixels
fixXY = [[[-1 1] * m / 2 0 0] + p.ScreenRect(3) / 2;
         [0 0 [-1 1] * m / 2] + p.ScreenRect(4) / 2];
nDispSize = length(p.dispSize);
nTrials = repeats * nDispSize * 2; % total number of trials
p.randSeed = ClockRandSeed;   % use clock to set the seed
                        % for the random number generator
radius = p.radius * ppd;      % radius in pixels
theta = (0 : 360/15 : 359)';  % polar angles of the 15 
                              % locations
[x y] = meshgrid(linspace(-1, 1, m));
r = sqrt(x .^ 2  + y .^ 2);
circle = (1 - exp(-((r - 0.85) / 0.15) .^ 4)) * p.ScreenBackground;     % circle with blurred edge
texB = Screen('MakeTexture', windowPtr, ones(m) * p.ScreenBackground, 0, 0, 2); % background
texD = Screen('MakeTexture', windowPtr, circle, 0, 0, 2); 
                        % distractors
texT = Screen('MakeTexture', windowPtr, circle, 0, 0, 2); 
                        % target
tex = texD;
if nargin && C_in_Os, tex = texT; end
Screen('FillRect', tex, p.ScreenBackground, [m/2 m/4 ...
       m m/4*3]);       % 'C' open to right

% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'sizeIndex' 'targetPresent' ...   
      'respCorrect' 'respTime'};
rec = nan(nTrials, length(p.recLabel)); 
        % matrix rec initialized with NaN
rec(:, 1) = 1 : nTrials;% count the trial number from 1 
                        % to nTrials
sizeIndex = repmat(1 : nDispSize, [2 1 repeats]);
targetPresent = zeros(2, nDispSize, repeats); 
                        % first set all to 0
targetPresent(:, :, 1 : repeats / 2) = 1; 
                        % change first half to 1
[rec(:, 2) ind] = Shuffle(sizeIndex(:)); 
                        % shuffle size index
rec(:, 3) = targetPresent(ind); % shuffle target presence 
                                % index in the same order

% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));

% Start experiment with instructions
str = [ 'Press the left arrow for target absent, and the ' ...
       ' right arrow for target present reponses\n\n ' ...
       ' Please respond as quickly and accurately as '  ...
       ' possible.\n\n Press SPACE to start.'];
DrawFormattedText(windowPtr, str, 'center', 'center', 1); 
        % Draw Instruction text string
Screen('Flip', windowPtr); 
        % flip the text image into the active buffer
Beeper;
WaitTill('space');      % wait till space bar is pressed
Secs = Screen('Flip', windowPtr);
p.start = datestr(now); % record start time

% Run nTrials trials
for i = 1 : nTrials
    % parameters for this trial
    sz = p.dispSize(rec(i, 2)); 
        % display size: total # of items in the display
    angles = theta + rand * 360;
    x = radius * cosd(angles) + p.ScreenRect(3) / 2; 
        % center locations of the recs
    y = radius * sind(angles) + p.ScreenRect(4) / 2;
    x = x + (rand(15, 1) - 0.5) * 2 * p.jitter; 
        % apply jitter
    y = y + (rand(15, 1) - 0.5) * 2 * p.jitter;
    rects = CenterRectOnPoint([0 0 m m], x, y); 
        % 15 recs
    nLoc = sz + sz / 4 - 1;
    tex(1 : nLoc) = texD; % set up distractors
    tex([5 10 nLoc+1:15]) = texB; 
        % 5, 10 and higher ones are set as background
    if rec(i, 3)          % target present
        ind = randi(nLoc);
        if mod(ind, 5) == 0, ind = ind + 1; end 
                          % avoid blank location
        tex(ind) = texT;  % draw a target in a target 
                          % present trial
    end
    WaitTill(Secs + p.ITI);
    Screen('DrawLines', windowPtr, fixXY, 3, 0); % fixation
    t0 = Screen('Flip', windowPtr, 0, 1);
    Screen('DrawTextures', windowPtr, tex, [], rects'); 
        % C and O
    t0 = Screen('Flip', windowPtr, t0 + p.fixDuration);
    [key Secs] = WaitTill(keys);  % wait till response
    Screen('Flip', windowPtr);    % remove stimulus
    if iscellstr(key), key = key{1}; end 
        % take the first key press in case of multiple 
        % key presses
    if strcmp(key, 'esc'), break; end
    rec(i, 4) = strcmp(key, 'right') == rec(i, 3); 
        % record response accuracy
    rec(i, 5) = Secs - t0;        % record respTime
    if rec(i, 4), Beeper; end     % beep if correct
end
p.finish = datestr(now);          % record finish time
save RT_rst.mat rec p;            % save the results


%% System Reinstatement Module

Priority(0);  % restore priority
sca; % close window and textures, restore color lookup table


